#!/bin/sh

TARGET_UUID="cc202507-0001-4abc-8aaa-000000000001"
HotateCache_GUID="514DCD5B-7899-45C9-88BB-8AF78FC12908"

# Full path to commands
SFDISK="/usr/local/sbin/sfdisk"
LS="/bin/ls -1"
BLKID="/sbin/blkid"
LSBLK="/usr/local/bin/lsblk"

# Log file path
LOG_FILE="/tmp/hf_extra_set_guid_ntfs.log"

# Logging functions
log() {
  echo "[INFO] $1" | tee -a "$LOG_FILE"
}

error() {
  echo "[ERROR] $1" | tee -a "$LOG_FILE" >&2
}

# Detect Hotate disk (by Partition Type GUID)
find_disk_by_partition_type_guid() {
  for dev in $(${LS} /dev/sd? /dev/nvme?n? 2>/dev/null); do
    if sudo ${SFDISK} -d "$dev" 2>/dev/null | grep -iq "type=$HotateCache_GUID"; then
      echo "$dev"
      return 0
    fi
  done
  return 1
}

# Detect NTFS partition (using blkid)
find_ntfs_partition() {
  disk="$1"
  for partname in $(${LSBLK} -nlo NAME "$disk"); do
    partpath="/dev/$partname"
    fstype=$(${BLKID} -s TYPE -o value "$partpath")
    if [ "$fstype" = "ntfs" ]; then
      echo "$disk" "$partname"
      return 0
    fi
  done
  return 1
}

# Start execution
log "Starting GUID setting script."

# Detect Hotate disk
DEVICE=$(find_disk_by_partition_type_guid)
if [ -z "$DEVICE" ]; then
  error "No disk found containing partition with HotateCache_GUID."
  exit 1
fi

# Identify NTFS partition
candidates=$(find_ntfs_partition "$DEVICE")

if [ -z "$candidates" ]; then
  error "No NTFS partition found."
  exit 1
fi

# Extract partition name and number
partname=$(echo "$candidates" | awk '{print $2}')
partnum="${partname##*[!0-9]}"

if [ -z "$partnum" ]; then
  error "Failed to extract partition number."
  exit 1
fi

log "Target device: $DEVICE, Partition: $partname (Number: $partnum)"
log "Setting GUID: $TARGET_UUID"

sudo ${SFDISK} --part-uuid "$DEVICE" "$partnum" "$TARGET_UUID"
log "GUID setting complete."
